/*
 * timer_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * The vHWTimerTask task configures the Timer 0 peripheral of the TM4C123GH6PM
 * MCU to run in split pair mode with Timer A configured for a capture event.
 * The capture event is set to be a rising / positive edge trigger on pin PF0.
 * The timer is loaded with 10 as the initial count and the match is set to 0.
 *
 * After the timer is enabled, it will monitor the PF0 pin and each rising edge
 * will decrement the load count until it hit zero.  At that point, an
 * interrupt will be fired.
 *
 * PF0 was chosen as it is the SW2 switch on the EK-TM4C123GXL LaunchPad.
 *
 * prvDataOutputTask will periodically monitor the Timer value register to see
 * if the count has changed.  If it has, it will print a UART message that
 * indicates such.
 *
 * This example uses UARTprintf for output of UART messages.  UARTprintf is not
 * a thread-safe API and is only being used for simplicity of the demonstration
 * and in a controlled manner.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_timer.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/timer.h"
#include "driverlib/uart.h"
#include "utils/uartstdio.h"

/*-----------------------------------------------------------*/

/*
 * Declare a variable that is used to hold the handle of the timer
 * interrupt task.
 */
TaskHandle_t xTimerIntTask = NULL;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvTimerIntTask( void *pvParameters );
static void prvDataOutputTask( void *pvParameters );

/*
 * Called by main() to create the hardware timer task.
 */
void vHWTimerTask( void );

/* 
 * Configure the hardware timer 0A on the TM4C123GH6PM to run in capture-
 * compare mode with a match interrupt.
*/
static void prvConfigureHWTimer( void );
/*-----------------------------------------------------------*/

void vHWTimerTask( void )
{
    /* Configure the hardware timer to run in capture-compare mode. */
    prvConfigureHWTimer();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for data output - for debug only as it is not used by
     *    the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvDataOutputTask,
                 "Output1",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 NULL );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for Timer 0A - for debug only as it is not used by
     *    the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvTimerIntTask,
                 "Timer0A",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 2,
                 &xTimerIntTask );

    /* Tasks have been created, now enable Timer 0A.
     * Depending on the application, this can be placed inside of the Configure
     * Hardware function too.  For this example, the task to receive the ISR
     * notification needs to be created first, so the Timer is only enabled
     * after the task has been created. */
    TimerEnable(TIMER0_BASE, TIMER_A);
}
/*-----------------------------------------------------------*/

static void prvDataOutputTask( void *pvParameters )
{
/* Counters to output how many button presses have occurred. */
uint32_t ui32HisorticCount, ui32CurrentCount;

    /* Set initial count value based on what has been loaded into the timer. */
    ui32HisorticCount = TimerValueGet(TIMER0_BASE, TIMER_A);

    /* Print the initial count. */
    UARTprintf("Countdown: %2d\r", ui32HisorticCount);

    for( ;; )
    {
        /* Enter a blocked state until 200 ms passes to delay initial update. */
        vTaskDelay( 200 );

        /* Get the current timer count. */
        ui32CurrentCount = TimerValueGet(TIMER0_BASE, TIMER_A);

        /* Has the timer count changed since it was last checked? */
        if(ui32CurrentCount != ui32HisorticCount)
        {
            /* Yes - update the display. */
            UARTprintf("Countdown: %2d\r", ui32CurrentCount);

            /* Update to reflect the new count value. */
            ui32HisorticCount = ui32CurrentCount;
        }
    }
}
/*-----------------------------------------------------------*/

static void prvTimerIntTask( void *pvParameters )
{
unsigned long ulEventsToProcess;
/* Interrupt counter indicating how often the timer down counter reached 0. */
uint32_t ui32IntCount = 0;

    for( ;; )
    {
        /* Wait to receive a notification sent directly to this task from the
        interrupt service routine. */
        ulEventsToProcess = ulTaskNotifyTake( pdTRUE, portMAX_DELAY );

        if (ulEventsToProcess != 0)
        {
            /* Update the interrupt counter. */
            ui32IntCount++;

            /* The timer is automatically stopped when it reaches the match
             * value so re-enable it now. */
            TimerEnable(TIMER0_BASE, TIMER_A);

            /* Output the current interrupt count. */
            UARTprintf("\t\tNumber of interrupts: %d\r", ui32IntCount);
        }
        else
        {
            /* Timed out. */
        }
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureHWTimer( void )
{
    /* The Timer 0 peripheral must be enabled for use. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_TIMER0);

    /* Configure PF0 as the CCP0 pin for timer 0. */
    GPIOPinTypeTimer(GPIO_PORTF_BASE, GPIO_PIN_0);
    GPIOPinConfigure(GPIO_PF0_T0CCP0);

    /* Set the pin to use the internal pull-up. */
    GPIOPadConfigSet(GPIO_PORTF_BASE, GPIO_PIN_0, GPIO_STRENGTH_2MA,
                     GPIO_PIN_TYPE_STD_WPU);

    /* Configure Timer 0A for Capture-Compare Mode for downward event count. */
    TimerConfigure(TIMER0_BASE, (TIMER_CFG_SPLIT_PAIR | TIMER_CFG_A_CAP_COUNT));

    /* Set the Timer 0A to trigger on a positive edge */
    TimerControlEvent(TIMER0_BASE, TIMER_A, TIMER_EVENT_POS_EDGE);

    /* Set the Timer 0A load value to 10. */
    TimerLoadSet(TIMER0_BASE, TIMER_A, 10);

    /* Set the Timer 0A match value to 0. */
    TimerMatchSet(TIMER0_BASE, TIMER_A, 0);

    /* Clear the interrupt status flag.  This is done to make sure the
    interrupt flag is cleared before we enable it. */
    TimerIntClear(TIMER0_BASE, TIMER_CAPA_MATCH);

    /* Configure the Timer 0A interrupt for Capture-Compare match.  With the
     * downward mode count, the timer will start with a load value of 10 and
     * decrement the value each event until it hits 0.  Then the flag for a
     * match interrupt will be set and trigger the hardware interrupt. */
    TimerIntEnable(TIMER0_BASE, TIMER_CAPA_MATCH);

    /* Enable the Timer 0A interrupt in the NVIC. */
    IntEnable(INT_TIMER0A);
}
/*-----------------------------------------------------------*/

void xTimer0AHandler( void )
{
BaseType_t xHigherPriorityTaskWoken;

    /* Clear the interrupt. */
    TimerIntClear(TIMER0_BASE, TIMER_CAPA_MATCH);

    /* The xHigherPriorityTaskWoken parameter must be initialized to pdFALSE as
    it will get set to pdTRUE inside the interrupt safe API function if a
    context switch is required. */
    xHigherPriorityTaskWoken = pdFALSE;

    /* Send a notification directly to the task to which interrupt processing
    is being deferred. */
    vTaskNotifyGiveFromISR( xTimerIntTask, &xHigherPriorityTaskWoken );

    /* This FreeRTOS API call will handle the context switch if it is required
    or have no effect if that is not needed. */
    portYIELD_FROM_ISR( xHigherPriorityTaskWoken );
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
    /* This function will be called by each tick interrupt if
        configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
        added here, but the tick hook is called from an interrupt context, so
        code must not attempt to block, and only the interrupt safe FreeRTOS API
        functions can be used (those that end in FromISR()). */

    /* Only the full demo uses the tick hook so there is no code is
        executed here. */
}


